<?php
require_once 'admin_config.php';
requireAdminAuth(); // Ensure only admins can access this page

$service = null;
$success = false;
$error = '';

// Get service ID from URL
$service_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Fetch existing service data
if ($service_id > 0) {
    try {
        $conn = getAdminDBConnection();
        $stmt = $conn->prepare("SELECT id, title, description, price, features, popular, category, gradient, icon FROM services WHERE id = ?");
        if ($stmt === false) {
            throw new Exception('Prepare fetch statement failed: ' . $conn->error);
        }
        $stmt->bind_param('i', $service_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $service = $result->fetch_assoc();
        } else {
            $error = 'الخدمة المطلوبة غير موجودة.';
        }
        $stmt->close();
    } catch (Exception $e) {
        $error = 'خطأ أثناء تحميل بيانات الخدمة: ' . $e->getMessage();
    }
} else {
    $error = 'معرف الخدمة غير صالح.';
}

// Handle form submission for updating a service
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $service_id > 0 && $service) {
    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $price = trim($_POST['price'] ?? '');
    $features = trim($_POST['features'] ?? '');
    $popular = isset($_POST['popular']) ? 1 : 0;
    $category = trim($_POST['category'] ?? '');
    $gradient = trim($_POST['gradient'] ?? '');
    $icon = trim($_POST['icon'] ?? '');

    if (empty($title) || empty($price) || empty($category)) {
        $error = 'يرجى ملء الحقول المطلوبة (الاسم، السعر، الفئة).';
    } else {
        try {
            $conn = getAdminDBConnection();

            $stmt = $conn->prepare("UPDATE services SET title = ?, description = ?, price = ?, features = ?, popular = ?, category = ?, gradient = ?, icon = ?, updated_at = NOW() WHERE id = ?");
            if ($stmt === false) {
                throw new Exception('Prepare update statement failed: ' . $conn->error);
            }

            // Bind parameters based on their types
            // s: string, d: double, i: integer
            $stmt->bind_param('ssdssissi', $title, $description, $price, $features, $popular, $category, $gradient, $icon, $service_id);

            if ($stmt->execute()) {
                $message = 'تم تحديث الخدمة بنجاح!';
                $message_type = 'success';
                header('Location: admin_services.php?message=' . urlencode($message) . '&type=' . $message_type);
                exit();
            } else {
                $error = 'حدث خطأ أثناء تحديث الخدمة: ' . $conn->error;
            }
            $stmt->close();
        } catch (Exception $e) {
            $error = 'خطأ: ' . $e->getMessage();
        }
    }
}

require_once '../header.php'; // Include header for consistent styling
?>

<div class="container bg-dark text-white mt-5">
    <div class="card">
        <div class="card-header">
            <h3 class="mb-0">تعديل الخدمة: <?= htmlspecialchars($service['title'] ?? '') ?></h3>
        </div>
        <div class="card-body">
            <?php if ($success): ?>
                <div class="alert alert-success" role="alert">تم تحديث الخدمة بنجاح!</div>
            <?php elseif ($error): ?>
                <div class="alert alert-danger" role="alert">
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <?php if ($service): ?>
            <form method="POST">
                <div class="mb-3">
                    <label for="title" class="form-label">اسم الخدمة <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="title" name="title" value="<?= htmlspecialchars($service['title'] ?? '') ?>" required>
                </div>
                <div class="mb-3">
                    <label for="description" class="form-label">الوصف</label>
                    <textarea class="form-control" id="description" name="description" rows="3"><?= htmlspecialchars($service['description'] ?? '') ?></textarea>
                </div>
                <div class="mb-3">
                    <label for="price" class="form-label">السعر <span class="text-danger">*</span></label>
                    <input type="number" step="0.01" class="form-control" id="price" name="price" value="<?= htmlspecialchars($service['price'] ?? '') ?>" required>
                </div>
                <div class="mb-3">
                    <label for="features" class="form-label">المميزات (افصلها بفاصلة)</label>
                    <input type="text" class="form-control" id="features" name="features" value="<?= htmlspecialchars($service['features'] ?? '') ?>">
                </div>
                <div class="mb-3 form-check">
                    <input type="checkbox" class="form-check-input" id="popular" name="popular" <?= ($service['popular'] ?? false) ? 'checked' : '' ?>>
                    <label class="form-check-label" for="popular">شائع</label>
                </div>
                <div class="mb-3">
                    <label for="category" class="form-label">الفئة <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="category" name="category" value="<?= htmlspecialchars($service['category'] ?? '') ?>" required>
                </div>
                <div class="mb-3">
                    <label for="gradient" class="form-label">التدرج اللوني (Gradient)</label>
                    <input type="text" class="form-control" id="gradient" name="gradient" placeholder="مثال: linear-gradient(90deg, #e14eca 0%, #7367f0 100%)" value="<?= htmlspecialchars($service['gradient'] ?? '') ?>">
                </div>
                <div class="mb-3">
                    <label for="icon" class="form-label">أيقونة (Bootstrap Icon Class)</label>
                    <input type="text" class="form-control" id="icon" name="icon" placeholder="مثال: bi-facebook" value="<?= htmlspecialchars($service['icon'] ?? '') ?>">
                </div>
                <button type="submit" class="btn btn-primary">تحديث الخدمة</button>
                <a href="admin_services.php" class="btn btn-secondary">إلغاء</a>
            </form>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php require_once '../footer.php'; ?> 